/*******************************************************
 HIDAPI - Multi-Platform library for
 communication with HID devices.

 Alan Ott
 Signal 11 Software

 8/22/2009

 Copyright 2009, All Rights Reserved.

 At the discretion of the user of this library,
 this software may be licensed under the terms of the
 GNU Public License v3, a BSD-Style license, or the
 original HIDAPI license as outlined in the LICENSE.txt,
 LICENSE-gpl3.txt, LICENSE-bsd.txt, and LICENSE-orig.txt
 files located at the root of the source distribution.
 These files may also be found in the public source
 code repository located at:
        http://github.com/signal11/hidapi .
********************************************************/

/** @file
 * @defgroup API hidapi API
 */

#ifndef HIDAPI_H__
#define HIDAPI_H__

/* ignore GCC warning
 *  wchar.h:392:1: warning: comparison of unsigned expression >= 0 is always true [-Wtype-limits]
 * */
#pragma GCC diagnostic ignored "-Wtype-limits"
#include <wchar.h>
#include <stdint.h>
#include <libusb-1.0/libusb.h>

#ifdef _WIN32
      #define HID_API_EXPORT __declspec(dllexport)
      #define HID_API_CALL
#else
      #define HID_API_EXPORT /**< API export macro */
      #define HID_API_CALL /**< API call macro */
#endif

#define HID_BUFF_MAX    1024    /* USB packet maximum */
#define HID_API_EXPORT_CALL HID_API_EXPORT HID_API_CALL /**< API export and call macro*/

#define HID_API_OK      0       /* API OK */
#define HID_API_NG      -1      /* API NG */
#define HID_API_NOT_CON -2      /* device not connected */

#define HID_STR_DESC_LANG_ID_LEN    4       /* length for strings descriptor language ID */
#define HID_REP_DESC_SITEM_SZ_MASK  0x3     /* bit mask for report descriptor short item size bit */
#define HID_KERNEL_DRV_ACTIVE       1       /* kernel driver is active */
#define HID_KERNEL_DRV_INACTIVE     0       /* kernel driver is inactive */

#ifdef __cplusplus
extern "C" {
#endif
        struct iap2_hid_device_;
        typedef struct iap2_hid_device_ iap2_hid_device; /**< opaque hidapi structure */

        /** hidapi info structure */
        struct iap2_hid_device_info {
            /** Platform-specific device path */
            char *path;
            /** Device Vendor ID */
            unsigned short vendor_id;
            /** Device Product ID */
            unsigned short product_id;
            /** Serial Number */
            wchar_t *serial_number;
            /** Device Release Number in binary-coded decimal,
                also known as Device Version Number */
            unsigned short release_number;
            /** Manufacturer String */
            wchar_t *manufacturer_string;
            /** Product string */
            wchar_t *product_string;
            /** Usage Page for this Device/Interface
                (Windows/Mac only). */
            unsigned short usage_page;
            /** Usage for this Device/Interface
                (Windows/Mac only).*/
            unsigned short usage;
            /** The USB interface which this logical device
                represents. Valid on both Linux implementations
                in all cases, and valid on the Windows implementation
                only if the device contains more than one interface. */
            int interface_number;

            /** Pointer to the next device */
            struct iap2_hid_device_info *next;
        };

        /* Interface structure for vendor specific request */
        typedef struct _iap2_hid_vctrl
        {
            uint8_t         bmRequestType;
            uint8_t         bRequest;
            uint16_t        wValue;
            uint16_t        wIndex;
            unsigned char   *buffer;
            uint16_t        wLength;
            unsigned int    timeout;
        } iap2_hid_vctrl, *p_iap2_hid_vctrl;



        typedef struct _iap2_hid_poll_fd
        {
            int fd;
            short int events;
        } iap2_hid_poll_fd;


        /** @brief Calls libusb handle event function.

            This function calls the libusb handle event function in case
            one of the libusb file desciptors was triggered.
            Otherwise it reads the event file descriptor.

            @ingroup API
            @param

            @returns
                This function returns HID_API_OK on success or -1 on error.
        */
        int HID_API_EXPORT iap2_hid_libusb_handle_events(libusb_context *usb_context,iap2_hid_device *dev, int pollFD);

        /** @brief Get a set of libusb file descriptors.

            This function returns a list of libusb file desciptors and
            their respective events. Furthermore it returns the number
            of available file descriptors.

            @ingroup API
            @param

            @returns
                This function returns a structure of file descriptors on
                success or -1 on error.
        */
        int HID_API_EXPORT iap2_hid_libusb_get_pollfds(iap2_hid_device *device,libusb_context* usb_context,iap2_hid_poll_fd *pollFDs, int* fd_count);

        /** @brief Initialize the HIDAPI library.

            This function initializes the HIDAPI library. Calling it is not
            strictly necessary, as it will be called automatically by
            iap2_hid_enumerate() and any of the iap2_hid_open*() functions if it is
            needed.  This function should be called at the beginning of
            execution however, if there is a chance of HIDAPI handles
            being opened by different threads simultaneously.

            @ingroup API

            @returns
                This function returns 0 on success and -1 on error.
        */
        libusb_context* HID_API_EXPORT HID_API_CALL iap2_hid_init(void);

        /** @brief Finalize the HIDAPI library.

            This function frees all of the static data associated with
            HIDAPI. It should be called at the end of execution to avoid
            memory leaks.

            @ingroup API

            @returns
                This function returns 0 on success and -1 on error.
        */
        int HID_API_EXPORT HID_API_CALL iap2_hid_exit(libusb_context* usb_context);

        /** @brief Enumerate the HID Devices.

            This function returns a linked list of all the HID devices
            attached to the system which match vendor_id and product_id.
            If @p vendor_id and @p product_id are both set to 0, then
            all HID devices will be returned.

            @ingroup API
            @param vendor_id The Vendor ID (VID) of the types of device
                to open.
            @param product_id The Product ID (PID) of the types of
                device to open.

            @returns
                This function returns a pointer to a linked list of type
                struct #hid_device, containing information about the HID devices
                attached to the system, or NULL in the case of failure. Free
                this linked list by calling iap2_hid_free_enumeration().
        */
        struct iap2_hid_device_info HID_API_EXPORT * HID_API_CALL iap2_hid_enumerate(libusb_context* usb_context,unsigned short vendor_id, unsigned short product_id);

        /** @brief Free an enumeration Linked List

            This function frees a linked list created by iap2_hid_enumerate().

            @ingroup API
            @param devs Pointer to a list of struct_device returned from
                      iap2_hid_enumerate().
        */
        void  HID_API_EXPORT HID_API_CALL iap2_hid_free_enumeration(struct iap2_hid_device_info *devs);

        /** @brief Open a HID device using a Vendor ID (VID), Product ID
            (PID) and optionally a serial number.

            If @p serial_number is NULL, the first device with the
            specified VID and PID is opened.

            @ingroup API
            @param vendor_id The Vendor ID (VID) of the device to open.
            @param product_id The Product ID (PID) of the device to open.
            @param serial_number The Serial Number of the device to open
                               (Optionally NULL).

            @returns
                This function returns a pointer to a #hid_device object on
                success or NULL on failure.
        */
        HID_API_EXPORT iap2_hid_device * HID_API_CALL iap2_hid_open(libusb_context* usb_context,unsigned short vendor_id, unsigned short product_id, const wchar_t *serial_number);

        /** @brief Open a HID device by its path name.

            The path name be determined by calling iap2_hid_enumerate(), or a
            platform-specific path name can be used (eg: /dev/hidraw0 on
            Linux).

            @ingroup API
            @param path The path name of the device to open

            @returns
                This function returns a pointer to a #hid_device object on
                success or NULL on failure.
        */
        HID_API_EXPORT iap2_hid_device * HID_API_CALL iap2_hid_open_path(libusb_context* usb_context,const char *path);

        /** @brief Write an Output report to a HID device.

            The first byte of @p data[] must contain the Report ID. For
            devices which only support a single report, this must be set
            to 0x0. The remaining bytes contain the report data. Since
            the Report ID is mandatory, calls to iap2_hid_write() will always
            contain one more byte than the report contains. For example,
            if a hid report is 16 bytes long, 17 bytes must be passed to
            iap2_hid_write(), the Report ID (or 0x0, for devices with a
            single report), followed by the report data (16 bytes). In
            this example, the length passed in would be 17.

            iap2_hid_write() will send the data on the first OUT endpoint, if
            one exists. If it does not, it will send the data through
            the Control Endpoint (Endpoint 0).

            @ingroup API
            @param device A device handle returned from iap2_hid_open().
            @param data The data to send, including the report number as
                the first byte.
            @param length The length in bytes of the data to send.

            @returns
                This function returns the actual number of bytes written and
                -1 on error.
        */
        int  HID_API_EXPORT HID_API_CALL iap2_hid_write(iap2_hid_device *device, const unsigned char *data, size_t length);

        /** @brief Read an Input report from a HID device with timeout.

            Input reports are returned
            to the host through the INTERRUPT IN endpoint. The first byte will
            contain the Report number if the device uses numbered reports.

            @ingroup API
            @param device A device handle returned from iap2_hid_open().
            @param data A buffer to put the read data into.
            @param length The number of bytes to read. For devices with
                multiple reports, make sure to read an extra byte for
                the report number.
            @param milliseconds timeout in milliseconds or -1 for blocking wait.

            @returns
                This function returns the actual number of bytes read and
                -1 on error.
        */
        int HID_API_EXPORT HID_API_CALL iap2_hid_read_timeout(iap2_hid_device *dev, unsigned char *data, size_t length, int milliseconds);

        /** @brief Read an Input report from a HID device.

            Input reports are returned
            to the host through the INTERRUPT IN endpoint. The first byte will
            contain the Report number if the device uses numbered reports.

            @ingroup API
            @param device A device handle returned from iap2_hid_open().
            @param data A buffer to put the read data into.
            @param length The number of bytes to read. For devices with
                multiple reports, make sure to read an extra byte for
                the report number.

            @returns
                This function returns the actual number of bytes read and
                -1 on error.
        */
        int  HID_API_EXPORT HID_API_CALL iap2_hid_read(iap2_hid_device *device, unsigned char *data, size_t length);

        /** @brief Set the device handle to be non-blocking.

            In non-blocking mode calls to hid_read() will return
            immediately with a value of 0 if there is no data to be
            read. In blocking mode, iap2_hid_read() will wait (block) until
            there is data to read before returning.

            Nonblocking can be turned on and off at any time.

            @ingroup API
            @param device A device handle returned from iap2_hid_open().
            @param nonblock enable or not the nonblocking reads
             - 1 to enable nonblocking
             - 0 to disable nonblocking.

            @returns
                This function returns 0 on success and -1 on error.
        */
        int  HID_API_EXPORT HID_API_CALL iap2_hid_set_nonblocking(iap2_hid_device *device, int nonblock);

        /** @brief Send a Feature report to the device.

            Feature reports are sent over the Control endpoint as a
            Set_Report transfer.  The first byte of @p data[] must
            contain the Report ID. For devices which only support a
            single report, this must be set to 0x0. The remaining bytes
            contain the report data. Since the Report ID is mandatory,
            calls to iap2_hid_send_feature_report() will always contain one
            more byte than the report contains. For example, if a hid
            report is 16 bytes long, 17 bytes must be passed to
            iap2_hid_send_feature_report(): the Report ID (or 0x0, for
            devices which do not use numbered reports), followed by the
            report data (16 bytes). In this example, the length passed
            in would be 17.

            @ingroup API
            @param device A device handle returned from iap2_hid_open().
            @param data The data to send, including the report number as
                the first byte.
            @param length The length in bytes of the data to send, including
                the report number.

            @returns
                This function returns the actual number of bytes written and
                -1 on error.
        */
        int HID_API_EXPORT HID_API_CALL iap2_hid_send_feature_report(iap2_hid_device *device, const unsigned char *data, size_t length);

        /** @brief Get a feature report from a HID device.

            Make sure to set the first byte of @p data[] to the Report
            ID of the report to be read.  Make sure to allow space for
            this extra byte in @p data[].

            @ingroup API
            @param device A device handle returned from iap2_hid_open().
            @param data A buffer to put the read data into, including
                the Report ID. Set the first byte of @p data[] to the
                Report ID of the report to be read.
            @param length The number of bytes to read, including an
                extra byte for the report ID. The buffer can be longer
                than the actual report.

            @returns
                This function returns the number of bytes read and
                -1 on error.
        */
        int HID_API_EXPORT HID_API_CALL iap2_hid_get_feature_report(iap2_hid_device *device, unsigned char *data, size_t length);

        /** @brief Close a HID device.

            @ingroup API
            @param device A device handle returned from iap2_hid_open().
        */
        void HID_API_EXPORT HID_API_CALL iap2_hid_close(libusb_context *usb_context, iap2_hid_device *device);

        /** @brief Get The Manufacturer String from a HID device.

            @ingroup API
            @param device A device handle returned from iap2_hid_open().
            @param string A wide string buffer to put the data into.
            @param maxlen The length of the buffer in multiples of wchar_t.

            @returns
                This function returns 0 on success and -1 on error.
        */
        int HID_API_EXPORT_CALL iap2_hid_get_manufacturer_string(iap2_hid_device *device, wchar_t *string, size_t maxlen);

        /** @brief Get The Product String from a HID device.

            @ingroup API
            @param device A device handle returned from iap2_hid_open().
            @param string A wide string buffer to put the data into.
            @param maxlen The length of the buffer in multiples of wchar_t.

            @returns
                This function returns 0 on success and -1 on error.
        */
        int HID_API_EXPORT_CALL iap2_hid_get_product_string(iap2_hid_device *device, wchar_t *string, size_t maxlen);

        /** @brief Get The Serial Number String from a HID device.

            @ingroup API
            @param device A device handle returned from iap2_hid_open().
            @param string A wide string buffer to put the data into.
            @param maxlen The length of the buffer in multiples of wchar_t.

            @returns
                This function returns 0 on success and -1 on error.
        */
        int HID_API_EXPORT_CALL iap2_hid_get_serial_number_string(iap2_hid_device *device, wchar_t *string, size_t maxlen);

        /** @brief Get a string from a HID device, based on its string index.

            @ingroup API
            @param device A device handle returned from iap2_hid_open().
            @param string_index The index of the string to get.
            @param string A wide string buffer to put the data into.
            @param maxlen The length of the buffer in multiples of wchar_t.

            @returns
                This function returns 0 on success and -1 on error.
        */
        int HID_API_EXPORT_CALL iap2_hid_get_indexed_string(iap2_hid_device *device, int string_index, wchar_t *string, size_t maxlen);

#if 0
        /** @brief Get a string describing the last error which occurred.

            @ingroup API
            @param device A device handle returned from iap2_hid_open().

            @returns
                This function returns a string containing the last error
                which occurred or NULL if none has occurred.
        */

        HID_API_EXPORT const wchar_t* HID_API_CALL iap2_hid_error(iap2_hid_device *device);
#endif /* if 0 */

        /** The special function for ipod_ctrl */
        void* HID_API_EXPORT HID_API_CALL iAP2HidGetReportInformation(iap2_hid_device *dev);
        int HID_API_EXPORT HID_API_CALL iap2_hid_get_maximum_reportID(iap2_hid_device *dev);

#ifdef __cplusplus
}
#endif

#endif

